<?php
/** @noinspection PhpMultipleClassDeclarationsInspection */
declare(strict_types=1);

/**
 *	Volatile Memory Storage.
 *	Supports context.
 *	@category		Library
 *	@package		CeusMedia_Cache_Adapter
 *	@author			Christian Würker <christian.wuerker@ceusmedia.de>
 */
namespace CeusMedia\Cache\Adapter;

use CeusMedia\Cache\SimpleCacheInterface;
use CeusMedia\Cache\SimpleCacheInvalidArgumentException;
use CeusMedia\Common\Exception\Deprecation as DeprecationException;
use CeusMedia\Common\Net\HTTP\PartitionSession as HttpSession;
use CeusMedia\Common\Net\HTTP\Session as HttpPartitionSession;
use DateInterval;

/**
 *	Volatile Memory Storage.
 *	Supports context.
 *	@category		Library
 *	@package		CeusMedia_Cache_Adapter
 *	@author			Christian Würker <christian.wuerker@ceusmedia.de>
 */
class Session extends AbstractAdapter implements SimpleCacheInterface
{
	/**	@var	string|NULL				$encoder */
	protected ?string $encoder			= NULL;

	/**	@var	array					$enabledEncoders	List of allowed encoder classes */
	protected array $enabledEncoders	= [];

	/**	@var	HttpSession|HttpPartitionSession	$resource */
	protected HttpSession|HttpPartitionSession		$resource;

	/**
	 *	Constructor.
	 *	@access		public
	 *	@param		HttpSession|array|string	$resource		Session object or list of partition name (optional) and session name (default: sid) or string PARTITION[@SESSION]
	 *	@param		string|NULL					$context		Internal prefix for keys for separation
	 *	@param		integer|NULL				$expiration		Data lifetime in seconds or expiration timestamp
	 *	@return		void
	 */
	public function __construct( $resource, ?string $context = NULL, ?int $expiration = NULL )
	{
		if( $resource instanceof HttpSession )
			$this->resource	= $resource;
		else{
			if( is_string( $resource ) )
				$resource		= explode( "@", $resource );
			$partitionName	= $resource[0];
			$sessionName	= $resource[1] ?? 'sid';
			if( $partitionName )
				$this->resource		= new HttpPartitionSession( $partitionName, $sessionName );
			else
				$this->resource		= new HttpSession( $sessionName );
		}
		if( $context !== NULL )
			$this->setContext( $context );
		if( $expiration !== NULL )
			$this->setExpiration( $expiration );
	}

	/**
	 *	Wipes clean the entire cache's keys.
	 *
	 *	@access		public
	 *	@return		bool		True on success and false on failure.
	 */
	public function clear(): bool
	{
		if( NULL === $this->context || '' === $this->context ){
			$this->resource->clear();
			return TRUE;
		}
		/** @var array $map */
		$map	= $this->resource->getAll( $this->context );
		return $this->deleteMultiple( array_keys( $map ) );
	}

	/**
	 *	Delete an item from the cache by its unique key.
	 *
	 *	@access		public
	 *	@param		string		$key		The unique cache key of the item to delete.
	 *	@return		boolean		True if the item was successfully removed. False if there was an error.
	 *	@throws		SimpleCacheInvalidArgumentException		if the $key string is not a legal value.
	 */
	public function delete( string $key ): bool
	{
		$this->checkKey( $key );
		if( !$this->resource->has( $this->context.$key ) )
			return FALSE;
		$this->resource->remove( $this->context.$key );
		return TRUE;
	}

	/**
	 *	Not implemented, yet.
	 *	Originally: Deletes multiple cache items in a single operation.
	 *
	 *	@param		iterable	$keys		A list of string-based keys to be deleted.
	 *	@return		boolean		True if the items were successfully removed. False if there was an error.
	 *	@throws		SimpleCacheInvalidArgumentException		if any of the $keys are not a legal value
	 */
	public function deleteMultiple( iterable $keys ): bool
	{
		foreach( $keys as $key )
			$this->checkKey( (string) $key );
		foreach( $keys as $key )
			$this->delete( (string) $key );
		return TRUE;
	}

	/**
	 *	Deprecated alias of clear.
	 *	@access		public
	 *	@return		self
	 *	@deprecated	use clear instead
	 */
	public function flush(): self
	{
		$this->clear();
		return $this;
	}

	/**
	 *	Fetches a value from the cache.
	 *
	 *	@access		public
	 *	@param		string		$key		The unique key of this item in the cache.
	 *	@param		mixed		$default	Default value to return if the key does not exist.
	 *	@return		mixed		The value of the item from the cache, or $default in case of cache miss.
	 *	@throws		SimpleCacheInvalidArgumentException		if the $key string is not a legal value.
	 */
	public function get( string $key, mixed $default = NULL ): mixed
	{
		$this->checkKey( $key );
		if( $this->resource->has( $this->context.$key ) )
			return $this->decodeValue( $this->resource->get( $this->context.$key ) );
		return NULL;
	}

	/**
	 *	Not implemented, yet.
	 *	Originally: Obtains multiple cache items by their unique keys.
	 *
	 *	@param		iterable	$keys		A list of keys that can obtained in a single operation.
	 *	@param		mixed		$default	Default value to return for keys that do not exist.
	 *	@return		array<string,mixed>		A list of key => value pairs. Cache keys that do not exist or are stale will have $default as value.
	 *	@throws		SimpleCacheInvalidArgumentException		if $keys is neither an array nor a Traversable,
	 *														or if any of the $keys are not a legal value.
	 *	@todo		implement
	 */
	public function getMultiple( iterable $keys, mixed $default = NULL ): array
	{
		$list	= [];
		foreach( $keys as $key )
			$this->checkKey( $key );
		/** @var string $key */
		foreach( $keys as $key )
			$list[$key]	= $this->get( $key );
		return $list;
	}

	/**
	 * 	Determines whether an item is present in the cache.
	 *
	 *	NOTE: It is recommended that has() is only to be used for cache warming type purposes
	 *	and not to be used within your live applications operations for get/set, as this method
	 *	is subject to a race condition where your has() will return true and immediately after,
	 *	another script can remove it, making the state of your app out of date.
	 *
	 *	@access		public
	 *	@param		string		$key		The cache item key.
	 *	@return		boolean
	 *	@throws		SimpleCacheInvalidArgumentException		if the $key string is not a legal value.
	 */
	public function has( string $key ): bool
	{
		$this->checkKey( $key );
		return $this->resource->has( $this->context.$key );
	}

	/**
	 *	Returns empty list since there is no stored data.
	 *	@access		public
	 *	@return		array
	 */
	public function index(): array
	{
		if( NULL === $this->context || '' === $this->context )
			return $this->resource->getKeys();

		/** @var array $map */
		$map	= $this->resource->getAll( $this->context );
		return array_keys( $map );
	}

	/**
	 *	Deprecated alias of delete.
	 *	@access		public
	 *	@param		string		$key		Data pair key
	 *	@return		boolean
	 *	@deprecated	use delete instead
	 */
	public function remove( string $key ): bool
	{
		throw DeprecationException::create()
			->setMessage( 'Deprecated' )
			->setSuggestion( 'Use delete instead' );
//		return $this->delete( $key );
	}

	/**
	 *	Persists data in the cache, uniquely referenced by a key with an optional expiration TTL time.
	 *
	 *	@access		public
	 *	@param		string					$key		The key of the item to store.
	 *	@param		mixed					$value		The value of the item to store. Must be serializable.
	 *	@param		DateInterval|int|NULL	$ttl		Optional. The TTL value of this item. If no value is sent and
	 *													the driver supports TTL then the library may set a default value
	 *													for it or let the driver take care of that.
	 *	@return		boolean		True on success and false on failure.
	 *	@throws		SimpleCacheInvalidArgumentException		if the $key string is not a legal value.
	 */
	public function set( string $key, mixed $value, DateInterval|int $ttl = NULL ): bool
	{
		$this->checkKey( $key );
		$json	= $this->encodeValue( $value );
		return $this->resource->set( $this->context.$key, $json );
	}

	/**
	 *	Not implemented, yet.
	 *	Originally: Persists a set of key => value pairs in the cache, with an optional TTL.
	 *
	 *	@param		iterable				$values		A list of key => value pairs for a multiple-set operation.
	 *	@param		DateInterval|int|NULL	$ttl		Optional. The TTL value of this item. If no value is sent and
	 *													the driver supports TTL then the library may set a default value
	 *													for it or let the driver take care of that.
	 *	@return		bool		True on success and false on failure.
	 *	@throws		SimpleCacheInvalidArgumentException		if $values is neither an array nor a Traversable,
	 *														or if any of the $values are not a legal value.
	 */
	public function setMultiple(iterable $values, DateInterval|int $ttl = NULL ): bool
	{
		foreach( $values as $key => $value )
			$this->checkKey( (string) $key );
		foreach( $values as $key => $value )
			$this->set( (string) $key, $value );
		return TRUE;
	}
}
