<?php /** @noinspection PhpMultipleClassDeclarationsInspection */

/**
 *	Searchs for Files by given RegEx Pattern (as File Name) in Folder.
 *
 *	Copyright (c) 2007-2024 Christian Würker (ceusmedia.de)
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 *	@category		Library
 *	@package		CeusMedia_Common_FS_File
 *	@author			Christian Würker <christian.wuerker@ceusmedia.de>
 *	@copyright		2007-2024 Christian Würker
 *	@license		https://www.gnu.org/licenses/gpl-3.0.txt GPL 3
 *	@link			https://github.com/CeusMedia/Common
 */

namespace CeusMedia\Common\FS\File;

use DirectoryIterator;
use RegexIterator;
use RuntimeException;

/**
 *	Searchs for Files by given RegEx Pattern (as File Name) in Folder.
 *	@category		Library
 *	@package		CeusMedia_Common_FS_File
 *	@author			Christian Würker <christian.wuerker@ceusmedia.de>
 *	@copyright		2007-2024 Christian Würker
 *	@license		https://www.gnu.org/licenses/gpl-3.0.txt GPL 3
 *	@link			https://github.com/CeusMedia/Common
 *	@todo			Fix Error while comparing File Name to Current File with Path
 */
class RegexFilter extends RegexIterator
{
	/**	@var	int				$numberFound			Number of found Files */
	protected int $numberFound		= 0;

	/**	@var	int				$numberScanned			Number of scanned Files */
	protected int $numberScanned	= 0;

	/**	@var	string			$contentPattern	Regular Expression to match with File Content */
	private ?string $contentPattern	= NULL;

	/**
	 *	Constructor.
	 *	@access		public
	 *	@param		string		$path		Path to search in
	 *	@param		string		$filePattern	Regular Expression to match with File Name
	 *	@return		void
	 */
	public function __construct( string $path, string $filePattern, ?string $contentPattern = NULL )
	{
		if( !file_exists( $path ) )
			throw new RuntimeException( 'Path "'.$path.'" is not existing.' );
		$this->numberFound		= 0;
		$this->numberScanned	= 0;
		$this->contentPattern	= $contentPattern;
		parent::__construct(
			new DirectoryIterator( $path ),
			$filePattern
		);
	}

	/**
	 *	Filter Callback.
	 *	@access		public
	 *	@return		bool
	 */
	public function accept(): bool
	{
		$this->numberScanned++;
		if( !parent::accept() )
			return FALSE;
		$this->numberFound++;
		if( !$this->contentPattern )
			return TRUE;
		$filePath	= $this->current()->getPathname();
		$realPath	= realpath( $this->current()->getPathname() );
		if( $realPath )
			$filePath	= $realPath;
		$content	= Reader::load( $filePath );
		return preg_match( $this->contentPattern, $content );
	}

	/**
	 *	Returns Number of found Files.
	 *	@access		public
	 *	@return		int
	 *	@noinspection	PhpUnused
	 */
	public function getNumberFound(): int
	{
		return $this->numberFound;
	}

	/**
	 *	Returns Number of scanned Files.
	 *	@access		public
	 *	@return		int
	 *	@noinspection	PhpUnused
	 */
	public function getNumberScanned(): int
	{
		return $this->numberScanned;
	}

	/**
	 *	Resets inner Iterator and numbers.
	 *	@access		public
	 *	@return		void
	 */
	#[\ReturnTypeWillChange]
	public function rewind(): void
	{
		$this->numberFound		= 0;
		$this->numberScanned	= 0;
		parent::rewind();
	}
}
