<?php
/**
 *	Builds Todo Info Site File.
 *
 *	Copyright (c) 2008-2021 Christian Würker (ceusmedia.de)
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *	@category		Tool
 *	@package		CeusMedia_DocCreator_Builder_HTML_Site_Info
 *	@author			Christian Würker <christian.wuerker@ceusmedia.de>
 *	@copyright		2008-2021 Christian Würker
 *	@license		http://www.gnu.org/licenses/gpl-3.0.txt GPL 3
 */
namespace CeusMedia\DocCreator\Builder\HTML\Site\Info;

use CeusMedia\DocCreator\Builder\HTML\Site\Info\Abstraction as SiteInfoAbstraction;

use UI_HTML_Elements as HtmlElements;

/**
 *	Builds Todo Info Site File.
 *	@category		Tool
 *	@package		CeusMedia_DocCreator_Builder_HTML_Site_Info
 *	@author			Christian Würker <christian.wuerker@ceusmedia.de>
 *	@copyright		2008-2021 Christian Würker
 *	@license		http://www.gnu.org/licenses/gpl-3.0.txt GPL 3
 */
class Todos extends SiteInfoAbstraction
{
	protected $count	= 0;

	/**
	 *	Creates Todo Info Site File.
	 *	@access		public
	 *	@return		bool		Flag: file has been created
	 *	@todo		support Interfaces, too
	 */
	public function createSite(): bool
	{
		$content	= "";
		$todos		= array();
		foreach( $this->env->data->getFiles() as $fileId => $file ){
			foreach( $file->getClasses() as $classId => $class ){
				$classTodos		= array();
				$methodTodos	= array();

				$classUri	= "class.".$class->getId().".html";
				$classLink	= HtmlElements::Link( $classUri, $class->getName(), 'class' );

				if( $class->getTodos() ){
					foreach( $class->getTodos() as $todo )
						$classTodos[]	= HtmlElements::ListItem( $todo, 1, array( 'class' => "classItem" ) );
					$this->count	+= count( $classTodos );
				}

				foreach( $class->getMethods() as $methodName => $methodData ){
					if( !$methodData->getTodos() )
						continue;
					$list	= array();
					foreach( $methodData->getTodos() as $todo )
						$list[]		= HtmlElements::ListItem( $todo, 2, array( 'class' => "methodItem" ) );
					$methodList		= HtmlElements::unorderedList( $list, 2, array( 'class' => "methodList" ) );
					$this->count	+= count( $list );
					$methodUrl		= 'class.'.$class->getId().".html#class_method_".$methodName;
					$methodLink		= HtmlElements::Link( $methodUrl, $methodName, 'method' );
					$methodTodos[]	= HtmlElements::ListItem( $methodLink.$methodList, 1, array( 'class' => "method" ) );
				}
				if( !$classTodos && !$methodTodos )
					continue;

				$methodTodos	= HtmlElements::unorderedList( $methodTodos, 1, array( 'class' => "methods" ) );
				$classTodos		= HtmlElements::unorderedList( $classTodos, 1, array( 'class' => "classList" ) );
				$todos[$class->getName()]		= HtmlElements::ListItem( $classLink.$classTodos.$methodTodos, 0, array( 'class' => "class" ) );
			}
		}
		ksort( $todos );
		if( $todos ){
			$this->verboseCreation( 'todos' );

			$words	= isset( $this->env->words['todos'] ) ? $this->env->words['todos'] : array();
			$uiData	= array(
				'title'		=> $this->env->builder->title->getValue(),
				'key'		=> 'todos',
				'id'		=> 'info-todos',
				'topic'		=> isset( $words['heading'] ) ? $words['heading'] : 'todos',
				'content'	=> HtmlElements::unorderedList( $todos, 0, array( 'class' => "classes" ) ),
				'words'		=> $words,
				'footer'	=> $this->buildFooter(),
			);
			$template	= 'site/info/todos';
			$template	= $this->hasTemplate( $template ) ? $template : 'site/info/abstract';
			$content	= $this->loadTemplate( $template, $uiData );
			$this->saveFile( "todos.html", $content );
			$this->appendLink( 'todos.html', 'todos', count( $todos ) );
		}
		return count( $todos ) > 0;
	}
}
