<?php

use CeusMedia\Common\ADT\Collection\Dictionary;
use CeusMedia\Common\FS\File\Reader as FileReader;
use CeusMedia\Common\FS\File\RegexFilter as RegexFileFilter;
use CeusMedia\HydrogenFramework\Controller;
use CeusMedia\HydrogenFramework\Environment;

class Controller_Blog extends Controller{

	/**	@var	Model_Article		$model		Article model instance */
	protected $model;
	/**	@var	boolean				$isEditor	Indicates wheter user is editor */
	protected $isEditor;

	/**
	 *	Constructor.
	 *	@access		public
	 *	@param		Environment		$env		Environment object
	 *	@return		void
	 */
	public function __construct( Environment $env ){
		parent::__construct( $env );
		$config			= $env->getConfig();
		$roleId			= $env->getSession()->get( 'roleId');
		$this->model	= new Model_Article( $env );
		$this->isEditor	= $roleId && $this->env->getAcl()->hasRight( $roleId, 'blog', 'add' );
	}

	/**
	 *	Adds an article, therefore receives title and content.
	 *	@access		public
	 *	@return		void
	 *	@throws		InvalidArgumentException	if title or content is missing
	 */
	public function add(){
		$request	= $this->env->getRequest();
		$messenger	= $this->env->getMessenger();
		$userId		= $this->env->getSession()->get( 'auth_user_id' );
		if( $request->get( 'do' ) == 'save' ){
			if( !strlen( trim( $request->get( 'title' ) ) ) )
				$messenger->noteError( 'Der Titel des Artikels fehlt.' );
			if( !strlen( trim( $request->get( 'content' ) ) ) )
				$messenger->noteError( 'Der Inhalt des Artikels fehlt.' );
			if( !$messenger->gotError() ){
				$data	= array(
					'title'			=> trim( $request->get( 'title' ) ),
					'content'		=> trim( $request->get( 'content' ) ),
					'createdAt'		=> time(),
					'status'		=> 0,
				);
				$model	= new Model_Article( $this->env );
				$articleId	= $model->add( $data, FALSE );
				if( !$articleId ){
					$this->env->getMessenger()->noteError( 'Der Artikel konnte nicht gespeichert werden.' );
					return;
				}
				$model	= new Model_ArticleAuthor( $this->env );
				$model->add( ['articleId' => $articleId, 'userId' => $userId] );
				$this->env->getMessenger()->noteSuccess( 'Der Artikel wurde gespeichert.' );
				$this->restart( './blog/edit/'.$articleId );
			}
		}
	}

	public function addAuthor( $articleId, $userId ){
		$request		= $this->env->getRequest();
		$modelRelation	= new Model_ArticleAuthor( $this->env );
		$indices		= ['articleId' => $articleId, 'userId' => $userId];
		if( !$modelRelation->getByIndices( $indices ) )
			$modelRelation->add( $indices );
		$this->restart( './blog/edit/'.$articleId );
	}

	public function addTag( $articleId ){
		$request		= $this->env->getRequest();
		$modelTag		= new Model_Tag( $this->env );
		$modelRelation	= new Model_ArticleTag( $this->env );

		if( (int) $articleId < 1 )
			$this->restart( './blog' );

		$tagName	= $request->get( 'tag' );
		if( !strlen( $tagName ) )
			$this->restart( './blog/edit/'.$articleId );

		$tags	= explode( ' ', str_replace( ',', ' ', $tagName ) );
		foreach( $tags as $tagName ){
			$tag		= $modelTag->getByIndex( 'title', trim( $tagName ) );
			if( $tag ){
				$tagId	= $tag->tagId;
				$number	= $tag->number;
				$indices	= [
					'articleId'	=> $articleId,
					'tagId'		=> $tagId,
				];
				if( $modelRelation->getByIndices( $indices ) )
					$this->restart( './blog/edit/'.$articleId );
			}
			else{
				$tagId	= $modelTag->add( array( 'title' => trim( $tagName ) ) );
				$number	= 0;
			}
			$data	= [
				'articleId'	=> $articleId,
				'tagId'		=> $tagId,
			];
			$modelRelation->add( $data );
			$modelTag->edit( $tagId, ['number' => ++$number] );
		}
		$this->restart( './blog/edit/'.$articleId );
	}

	public function article( $articleId, $version = 0, $title = NULL ){
		$articleId	= preg_replace( "/^(\d+).*/", '\\1', trim( $articleId ) );
		if( (int) $articleId < 1 )
			$this->restart( './blog' );

		$version	= preg_replace( "/^(\d+).*/", '\\1', trim( $version ) );

		$states		= $this->env->getSession()->get( 'filter_blog_states' );
		if( !$this->isEditor )
			$states	= [1];
		else if( !$states )
			$this->env->getSession()->set( 'filter_blog_states', $states = [0, 1] );
		$conditions	= ['status' => $states];

		$article				= $this->model->get( $articleId );
		$article->versions		= $this->model->getArticleVersions( $articleId );
		$article->version		= count( $article->versions ) + 1;
		if( $version < 1 )
			$version	= $article->version;

		$data	= array(
			'articles'	=> $this->model->getAll( $conditions, ['createdAt' => 'DESC'] ),
			'article'	=> $article,
			'tags'		=> $this->model->getArticleTags( $articleId ),
			'authors'	=> $this->model->getArticleAuthors( $articleId ),
			'articleId'	=> rawurldecode( $articleId ),
			'version'	=> $version,
			'config'	=> new Dictionary( $this->env->getConfig()->getAll( 'module.blog_compact.' ) )
		);
		$this->setData( $data );
	}

	public function author( $username = NULL ){
		$model	= new Model_User( $this->env );
		$user	= $model->getByIndex( 'username', $username );
		if( !$user ){
			$this->env->getMessenger()->noteError( 'Invalid user' );
			$this->restart( NULL, TRUE );
		}
		$this->addData( 'user', $user );
	}

	public function dev( $topic = NULL ){
		$topic		= strlen( $topic ) ? $topic : NULL;

		$index	= new RegexFileFilter( 'contents/dev/', "/\.txt$/" );
		$list	= [];
		foreach( $index as $file ){
			$fileName	= pathinfo( $file->getFilename(), PATHINFO_FILENAME );
			$list[$fileName]	= filemtime( $file->getPathname() );
		}
		$this->addData( 'files', $list );
		$this->addData( 'content', '' );

		if( $topic ){
			$fileName	= 'contents/dev/'.$topic.'.txt';
			if( file_exists( $fileName ) )
				$this->addData( 'content', FileReader::load( $fileName ) );
		}
	}

	public function edit( $articleId, $version = 0 ){
		$request	= $this->env->getRequest();
		$messenger	= $this->env->getMessenger();

		if( (int) $articleId < 1 )
			$this->restart( './blog' );
		$article	= $this->model->get( $articleId );
		if( !$article )
			$this->restart( './blog' );

		$model		= new Model_ArticleVersion( $this->env );
		$article->versions	= $this->model->getArticleVersions( $articleId );
		$article->version	= count( $article->versions ) + 1;
		if( $version < 1 )
			$version	= $article->version;

		$userId		= $this->env->getSession()->get( 'auth_user_id' );

		if( $request->get( 'save' ) ){
			if( !strlen( trim( $request->get( 'title' ) ) ) )
				throw new InvalidArgumentException( 'Article title is missing' ) ;
			if( !strlen( trim( $request->get( 'content' ) ) ) )
				throw new InvalidArgumentException( 'Article content is missing' ) ;

			$createdAt	= strtotime( $request->get( 'date' ).' '.$request->get( 'time' ).':00' );	//  new creation date string
			$createdAt	= ( $createdAt && !$request->get( 'now' ) ) ? $createdAt : time();

			$data		= array(
				'title'		=> trim( $request->get( 'title' ) ),
				'content'	=> trim( $request->get( 'content' ) ),
				'status'	=> $request->get( 'status' ),
				'createdAt'	=> $createdAt,
			);

			$model	= new Model_Article( $this->env );
			if( !$model->edit( $articleId, $data, FALSE ) ){
				$messenger->noteError( 'Am Artikel wurde nichts geändert.' );
				return;
			}
			$model->edit( $articleId, array( 'modifiedAt' => time() ) );
			$this->env->getMessenger()->noteSuccess( 'Der Artikel wurde geändert.' );

			if( $request->get( 'version' ) ){
				$model	= new Model_ArticleVersion( $this->env );
				$data	= array(
					'articleId'		=> $article->articleId,
					'title'			=> $article->title,
					'content'		=> $article->content,
					'createdAt'		=> $article->createdAt,
					'modifiedAt'	=> max( $article->createdAt, $article->modifiedAt ),
				);
				$model->add( $data, FALSE );
				$messenger->noteNotice( 'Die vorherige Version des Artikels wurde archiviert.' );
			}

			$model	= new Model_ArticleAuthor( $this->env );
			if( !$model->count( ['articleId' => $articleId, 'userId' => $userId] ) )
				$model->add( ['articleId' => $articleId, 'userId' => $userId] );


			$this->restart( './blog/edit/'.$articleId );
		}
		$modelUser	= new Model_User( $this->env );
		$data	= array(
			'articles'	=> $this->model->getAll(),
			'article'	=> $this->model->get( $articleId ),
			'tags'		=> $this->model->getArticleTags( $articleId ),
			'authors'	=> $this->model->getArticleAuthors( $articleId ),
			'editors'	=> $modelUser->getAll( ['status' => '> 0'] ),
			'articleId'	=> rawurldecode( $articleId ),
			'tags'		=> $this->model->getArticleTags( $articleId ),
			'version'	=> $version,
		);
		$this->setData( $data );
	}

	/**
	 *	Generates RSS Feed and returns it directly to the requesting client.
	 *	@access		public
	 *	@param		integer		$limit
	 *	@param		boolean		$debug
	 *	@return		void
	 */
	public function feed( $limit = 10, $debug = NULL ){
		$limit		= ( (int) $limit > 0 ) ? (int) $limit : 10;

		$conditions	= array( 'status' => 1, 'createdAt' => '<= '.time() );
		$orders		= ['createdAt' => 'DESC'];
		$articles	= $this->model->getAll( $conditions, $orders, [0, $limit] );
		$this->addData( 'articles', $articles );
		$this->addData( 'debug', (bool) $debug );
	}

	protected function getFilteredStates(){
		$states	= $this->env->getSession()->get( 'filter_blog_states' );
		if( !is_array( $states ) )
			$this->env->getSession()->set( 'filter_blog_states', $states = [1] );
		if( !$this->isEditor /*|| !$states*/ )
			$this->env->getSession()->set( 'filter_blog_states', $states = [1] );
		return $states;
	}

	public function index( $page = 0, $limit = NULL ){
		$perPage	= abs( (int) $this->env->getConfig()->get( 'module.blog_compact.perPage' ) );
		$states		= $this->getFilteredStates();

		$limit		= !is_null( $limit ) ? $limit : ( $perPage ? $perPage : 10 );
		$offset		= $page * $limit;
		$limits		= [$offset, $limit];
		$conditions	= ['status' => $states ? $states : -99];
		if( !$this->isEditor )
			$conditions['createdAt']	= '<=  '.time();
		$orders		= ['createdAt' => 'DESC'/*, 'articleId' => 'DESC'*/];
		$articles	= $this->model->getAll( $conditions, $orders, $limits );

		foreach( $articles as $nr => $article ){
			$article->authors		= $this->model->getArticleAuthors( $article->articleId );
			$article->tags			= $this->model->getArticleTags( $article->articleId );
			$article->versions		= $this->model->getArticleVersions( $article->articleId );
			$article->version		= count( $article->versions ) + 1;
		}

		$query		= 'SELECT COUNT(at.articleId) as nr, t.tagId, t.title FROM articles AS a, article_tags AS at, tags AS t WHERE at.tagId=t.tagId AND at.articleId=a.articleId AND a.status=1 GROUP BY t.tagId, t.title ORDER BY nr DESC LIMIT 0, 10';
		$topTags	= $this->env->getDatabase()->query( $query )->fetchAll( PDO::FETCH_OBJ );
		$data		= array(
			'page'		=> $page,
			'limit'		=> $limit,
			'offset'	=> $offset,
			'articles'	=> $articles,
			'states'	=> $states,
			'number'	=> $this->model->count( $conditions ),
			'topTags'	=> $topTags,
			'isEditor'	=> $this->isEditor,
			'config'	=> new Dictionary( $this->env->getConfig()->getAll( 'module.blog_compact.' ) )
		);
		$this->setData( $data );
	}

	public function removeAuthor( $articleId, $userId ){
		$model		= new Model_ArticleAuthor( $this->env );
		$model->removeByIndices( ['articleId' => $articleId, 'userId' => $userId] );
		$this->restart( './blog/edit/'.$articleId );
	}

	public function removeTag( $articleId, $tagId ){
		$model	= new Model_ArticleTag( $this->env );
		$indices	= [
			'articleId'	=> $articleId,
			'tagId'		=> $tagId,
		];
		if( $model->removeByIndices( $indices ) ){
			$model	= new Model_Tag( $this->env );
			$tag	= $model->get( $tagId );
			$number	= $tag->number--;
			$model->edit( $articleId, ['number' => $number] );
		}
		$this->restart( './blog/edit/'.$articleId );
	}

	public function setContent( $articleId ){
		$request	= $this->env->getRequest();
		if( (int) $articleId < 1 || !( $article	= $this->model->get( (int) $articleId ) ) )
			$this->restart( './blog' );

		if( $article->content != $request->get( 'content' ) ){
			$data	= array(
				'content'		=> $request->get( 'content' ),
				'modifiedAt'	=> time(),
			);
			$this->model->edit( $articleId, $data, FALSE );
		}
		$request->isAjax() ? exit : $this->restart( './blog/edit/'.$articleId );
	}

	public function setFilter(){
		$request	= $this->env->getRequest();
		$session	= $this->env->getSession();
		$mode		= $request->get( 'mode' );
		$name		= $request->get( 'name' );
		$value		= $request->get( 'value' );
		$store		= $session->get( 'filter_blog_'.$name );
#		$session->remove( 'filter_blog_states' );
		switch( $mode ){
			case 'add':
				if( !is_array( $store ) )
					$store	= [];
				$store[]	= $value;
				$session->set( 'filter_blog_'.$name, $store );
				break;
			case 'set':
				$session->set( 'filter_blog_'.$name, $value );
				break;
			case 'remove':
				if( !is_array( $store ) )
					$session->remove( 'filter_blog_'.$name );
				else{
					$store	= array_diff( $store, [$value] );
					$session->set( 'filter_blog_'.$name, $store );
				}
				break;
		}
		if( $request->isAjax() )
			exit;
		$this->restart( './blog' );
	}

	public function setStatus( $articleId, $status ){
		$words	= $this->getWords( 'states' );
		if( (int) $articleId < 1 )
			$this->restart( './blog' );
		$article	= $this->model->get( $articleId );
		if( !$article )
			$this->restart( './blog' );
		$this->model->edit( $articleId, ['status' => $status] );
		$this->env->getMessenger()->noteSuccess( 'Der Status wurde auf <cite>'.$words[$status].'</cite> gesetzt.' );
		$this->restart( './blog/edit/'.$articleId );
	}

	public function tag( $tagName ){
		$model		= new Model_Tag( $this->env );
		$tag		= $model->getByIndex( 'title', $tagName );
		$states		= $this->getFilteredStates();
		$articles		= [];
		$relatedTags	= [];
		if( $tag ){
			$model		= new Model_ArticleTag( $this->env );
			$relations	= $model->getAllByIndex( 'tagId', $tag->tagId );
			$model		= new Model_Article( $this->env );
			foreach( $relations as $relation ){
				$article	= $model->get( $relation->articleId );
				if( in_array( $article->status, $states ) )
					$articles[$article->articleId]	= $article;
			}
			krsort( $articles );

			$model			= new Model_Tag( $this->env );
			$relatedTags	= $model->getRelatedTags( $tag->tagId );
		}
		foreach( $articles as $nr => $article ){
			$articles[$nr]->authors	= $this->model->getArticleAuthors( $article->articleId );
			$articles[$nr]->tags	= $this->model->getArticleTags( $article->articleId );
		}
		$this->setData( [
			'articles'	=> $articles,
			'tag'		=> $tag,
			'tagName'	=> $tagName,
			'friends'	=> $relatedTags
		] );
	}

	public function thumb( $file ){
		$config	= $this->env->getConfig();
		$path	= $config->get( 'path.images' ).$config->get( 'module.blog_compact.path.images' );
		$this->addData( 'path', $path );
		$this->addData( 'file', $file );
	}
}
?>
